<?php
/**
 * Trait assets.
 *
 * @package   SQL_Buddy
 * @license   https://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @link      https://github.com/deliciousbrains/sql-buddy/
 */

namespace SQL_Buddy\Traits;

trait Assets {
	/**
	 * Get asset metadata.
	 *
	 * @param string $handle Script handle.
	 *
	 * @return array
	 */
	protected function get_asset_metadata( $handle ) {
		$base_path = SQL_BUDDY_PLUGIN_DIR_PATH . 'assets/js/';

		// *.asset.php is generated by DependencyExtractionWebpackPlugin.
		// *.chunks.php is generated by HtmlWebpackPlugin with a custom template.
		$asset_file  = $base_path . $handle . '.asset.php';
		$chunks_file = $base_path . $handle . '.chunks.php';
		$asset       = is_readable( $asset_file ) ? require $asset_file : [];
		$chunks      = is_readable( $chunks_file ) ? require $chunks_file : [];

		$asset['dependencies'] = isset( $asset['dependencies'] ) ? $asset['dependencies'] : [];
		// A hash calculated based on the file content of the entry point bundle at <$handle>.js.
		$asset['version'] = isset( $asset['version'] ) ? $asset['version'] : SQL_BUDDY_VERSION;
		$asset['js']      = isset( $chunks['js'] ) ? $chunks['js'] : [];
		$asset['css']     = isset( $chunks['css'] ) ? $chunks['css'] : [];

		return $asset;
	}

	/**
	 * Register script using handle.
	 *
	 * @param string $script_handle Handle of script.
	 * @param array  $script_dependencies Array of extra dependencies.
	 *
	 * @return void
	 */
	public function register_script( $script_handle, array $script_dependencies = [] ) {
		$base_script_path = SQL_BUDDY_PLUGIN_DIR_URL . 'assets/js/';
		$in_footer        = true;

		// Register any chunks of $script_handle first.
		$asset = $this->get_asset_metadata( $script_handle );
		// Webpack appends "-[contenthash]" to filenames of chunks, so omit the `?ver=` query param.
		$chunk_version = null;
		foreach ( $asset['js'] as $script_chunk ) {
			wp_register_script(
				$script_chunk,
				$base_script_path . $script_chunk . '.js',
				[],
				$chunk_version,
				$in_footer
			);
		}
		$dependencies = array_merge( $asset['dependencies'], $script_dependencies, $asset['js'] );

		$entry_version = $asset['version'];
		wp_register_script(
			$script_handle,
			$base_script_path . $script_handle . '.js',
			$dependencies,
			$entry_version,
			$in_footer
		);

		wp_set_script_translations( $script_handle, 'sql-buddy' );
	}

	/**
	 * Enqueue script using handle.
	 *
	 * @param string $script_handle Handle of script.
	 * @param array  $script_dependencies Array of extra dependencies.
	 *
	 * @return void
	 */
	public function enqueue_script( $script_handle, array $script_dependencies = [] ) {
		$this->register_script( $script_handle, $script_dependencies );
		wp_enqueue_script( $script_handle );
	}

	/**
	 * Register style using handle.
	 *
	 * @param string $style_handle Handle of script.
	 * @param array  $style_dependencies Array of extra dependencies.
	 *
	 * @return void
	 */
	public function register_style( $style_handle, array $style_dependencies = [] ) {
		$base_style_path = SQL_BUDDY_PLUGIN_DIR_URL . 'assets/css/';

		// Register any chunks of $style_handle first.
		$asset = $this->get_asset_metadata( $style_handle );
		// Webpack appends "-[contenthash]" to filenames of chunks, so omit the `?ver=` query param.
		$chunk_version = null;
		foreach ( $asset['css'] as $style_chunk ) {
			wp_register_style(
				$style_chunk,
				$base_style_path . $style_chunk . '.css',
				[],
				$chunk_version
			);
		}
		$style_dependencies = array_merge( $style_dependencies, $asset['css'] );

		$entry_version = $asset['version'];
		wp_register_style(
			$style_handle,
			$base_style_path . $style_handle . '.css',
			$style_dependencies,
			$entry_version
		);

		wp_style_add_data( $style_handle, 'rtl', 'replace' );
	}

	/**
	 * Enqueue style using handle.
	 *
	 * @param string $style_handle Handle of style.
	 * @param array  $style_dependencies Array of extra dependencies.
	 *
	 * @return void
	 */
	public function enqueue_style( $style_handle, array $style_dependencies = [] ) {
		$this->register_style( $style_handle, $style_dependencies );
		wp_enqueue_style( $style_handle );
	}
}
