<?php
/**
 *  This file is part of the Add-Meta-Tags distribution package.
 *
 *  Add-Meta-Tags is an extension for the WordPress publishing platform.
 *
 *  Homepage:
 *  - http://wordpress.org/plugins/add-meta-tags/
 *  Documentation:
 *  - http://www.codetrax.org/projects/wp-add-meta-tags/wiki
 *  Development Web Site and Bug Tracker:
 *  - http://www.codetrax.org/projects/wp-add-meta-tags
 *  Main Source Code Repository (Mercurial):
 *  - https://bitbucket.org/gnotaras/wordpress-add-meta-tags
 *  Mirror repository (Git):
 *  - https://github.com/gnotaras/wordpress-add-meta-tags
 *  Historical plugin home:
 *  - http://www.g-loaded.eu/2006/01/05/add-meta-tags-wordpress-plugin/
 *
 *  Licensing Information
 *
 *  Copyright 2006-2013 George Notaras <gnot@g-loaded.eu>, CodeTRAX.org
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  The NOTICE file contains additional licensing and copyright information.
 */


/**
 * Module containing settings related functions.
 */


/**
 * Returns an array with the default options.
 */
function amt_get_default_options() {
    return array(
        "settings_version"  => 4,       // IMPORTANT: SETTINGS UPGRADE: Every time settings are added or removed this has to be incremented.
        "site_description"  => "",      // Front page description
        "site_keywords"     => "",      // Front page keywords
        "global_keywords"   => "",      // These keywords are added to the 'keywords' meta tag on all posts and pages
        "site_wide_meta"    => "",
        "auto_description"  => "1",     // Descriptions auto-generated by default
        "auto_keywords"     => "1",     // Keywords auto-generated by default
        "auto_opengraph"    => "0",
        "auto_dublincore"   => "0",
        "auto_twitter"      => "0",
        "auto_schemaorg"    => "0",
        "noodp_description" => "0",
        "noindex_search_results"     => "1",
        "noindex_date_archives"      => "0",
        "noindex_category_archives"  => "0",
        "noindex_tag_archives"       => "0",
        "noindex_author_archives"    => "0",
        "copyright_url"     => "",
        "default_image_url" => "",
        "review_mode"       => "0",
        "i_have_donated"    => "0",
        );
}


/**
 * Performs upgrade of the plugin settings.
 */
function amt_plugin_upgrade() {

    // First we try to determine if this is a new installation or if the
    // current installation requires upgrade.

    // Default Add-Meta-Tags Settings
    $default_options = amt_get_default_options();

    // Try to get the current Add-Meta-Tags options from the database
    $stored_options = get_option("add_meta_tags_opts");
    if ( empty($stored_options) ) {
        // This is the first run, so set our defaults.
        update_option("add_meta_tags_opts", $default_options);
        return;
    }

    // Check the settings version

    // If the settings version of the default options matches the settings version
    // of the stored options, there is no need to upgrade.
    if (array_key_exists('settings_version', $stored_options) &&
            (intval($stored_options["settings_version"]) == intval($default_options["settings_version"])) ) {
        // Settings are up to date. No upgrade required.
        return;
    }

    // On any other case a settings upgrade is required.

    // 1) Add any missing options to the stored Add-Meta-Tags options
    foreach ($default_options as $opt => $value) {
        // Always upgrade the ``settings_version`` option
        if ($opt == 'settings_version') {
            $stored_options['settings_version'] = $value;
        }
        // Add missing options
        elseif ( !array_key_exists($opt, $stored_options) ) {
            $stored_options[$opt] = $value;
        }
        // Existing stored options are untouched here.
    }

    // 2) Migrate any current options to new ones.
    // Migration rules should go here.

    // Version 2.2.0 (settings_version 1->2)
    // Removed ``noindex_archives``
    // No migrations required. Clean-up takes place in step (3) below.

    // Version 2.2.1 (settings_version 2->3)
    // Added ``review_mode``
    // No migrations required. Addition takes place in (1).

    // Version 2.3.3 (settings_version 3->4)
    // Added ``auto_twitter``
    // Added ``auto_schemaorg``
    // No migrations required. Addition takes place in (1).


    // 3) Clean stored options.
    foreach ($stored_options as $opt => $value) {
        if ( !array_key_exists($opt, $default_options) ) {
            // Remove any options that do not exist in the default options.
            unset($stored_options[$opt]);
        }
    }

    // Finally save the updated options.
    update_option("add_meta_tags_opts", $stored_options);

}
add_action('plugins_loaded', 'amt_plugin_upgrade');


/**
 * Saves the new settings in the database.
 * Accepts the POST request data.
 */
function amt_save_settings($post_payload) {
    
    // Default Add-Meta-Tags Settings
    $default_options = amt_get_default_options();

    $add_meta_tags_opts = array();

    foreach ($default_options as $def_key => $def_value) {

        // **Always** use the ``settings_version`` from the defaults
        if ($def_key == 'settings_version') {
            $add_meta_tags_opts['settings_version'] = $def_value;
        }

        // Add options from the POST request (saved by the user)
        elseif ( array_key_exists($def_key, $post_payload) ) {

            // Validate and sanitize input before adding to 'add_meta_tags_opts'
            if ( $def_key == 'site_description' ) {
                $add_meta_tags_opts[$def_key] = sanitize_text_field( amt_sanitize_description( stripslashes( $post_payload[$def_key] ) ) );
            } elseif ( $def_key == 'site_keywords' ) {
                // No placeholders here
                $add_meta_tags_opts[$def_key] = sanitize_text_field( amt_sanitize_keywords( stripslashes( $post_payload[$def_key] ) ) );
            } elseif ( $def_key == 'global_keywords' ) {
                // placeholder may exist here
                $add_meta_tags_opts[$def_key] = amt_sanitize_keywords( amt_revert_placeholders( sanitize_text_field( amt_convert_placeholders( stripslashes( $post_payload[$def_key] ) ) ) ) );
            } elseif ( $def_key == 'site_wide_meta' ) {
                $add_meta_tags_opts[$def_key] = esc_textarea( wp_kses( stripslashes( $post_payload[$def_key] ), amt_get_allowed_html_kses() ) );
            } elseif ( $def_key == 'copyright_url' ) {
                $add_meta_tags_opts[$def_key] = esc_url_raw( stripslashes( $post_payload[$def_key] ), array( 'http', 'https') );
            } elseif ( $def_key == 'default_image_url' ) {
                $add_meta_tags_opts[$def_key] = esc_url_raw( stripslashes( $post_payload[$def_key] ), array( 'http', 'https') );
            } else {
                $add_meta_tags_opts[$def_key] = sanitize_text_field( stripslashes( $post_payload[$def_key] ) );
            }
        }
        
        // If missing (eg checkboxes), use the default value, except for the case
        // those checkbox settings whose default value is 1.
        else {

            // The following settings have a default value of 1, so they can never be
            // deactivated, unless the following check takes place.
            if (
                $def_key == 'auto_description' ||
                $def_key == 'auto_keywords' ||
                $def_key == 'noindex_search_results'
            ) {
                if( !isset($post_payload[$def_key]) ){
                    $add_meta_tags_opts[$def_key] = "0";
                }
            } else {
                // Else save the default value in the db.
                $add_meta_tags_opts[$def_key] = $def_value;
            }

        }
    }

    // Finally update the Add-Meta-Tags options.
    update_option("add_meta_tags_opts", $add_meta_tags_opts);

    //var_dump($post_payload);
    //var_dump($add_meta_tags_opts);

    amt_show_info_msg(__('Add-Meta-Tags options saved', 'add-meta-tags'));
}


/**
 * Reset settings to the defaults.
 */
function amt_reset_settings() {
    // Default Add-Meta-Tags Settings
    $default_options = amt_get_default_options();

    delete_option("add_meta_tags_opts");
    update_option("add_meta_tags_opts", $default_options);
    amt_show_info_msg(__('Add-Meta-Tags options were reset to defaults', 'add-meta-tags'));
}

